/** @file   MachineGun.cpp
 * @brief   Implementation of MachineGun class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "MachineGun.h"
#include "www_map.h"
#include "soundsamples.h"
#include "GfxManager.h"
#include "AnimPlayer.h"
#include "settings.h"
#include "GfxId.h"
#include "RedrawQueue.h"
#include "PlayerController.h"
#include "WarGlobals.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int MachineGun::KNoExitCounterIndex;
const int MachineGun::KReloadCounterIndex;
const int MachineGun::KGunNose1;
const int MachineGun::KGunNose2;



///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
MachineGun::MachineGun() :
  GameObject(),
  iMaster( 0 ),
  iActiveGunNose( 0 )
{
  int oid = ObjectID::TYPE_MACHINEGUN;
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  
  Vec2D nose1( Settings::floatObjProp(oid, "gun1_nose_x:"),
               Settings::floatObjProp(oid, "gun1_nose_y:") );
  Vec2D nose2( Settings::floatObjProp(oid, "gun2_nose_x:"),
               Settings::floatObjProp(oid, "gun2_nose_y:") );
  this->addCtrlPoint( nose1 );
  this->addCtrlPoint( nose2 );
}



/** Destructor.
 */
MachineGun::~MachineGun()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

void MachineGun::update()
{
  if ( iMaster == 0 ) {
    // No master object so we do nothing.
    return;
  }
  
  if ( iMaster->getController() == 0 || iMaster->objectType() != ObjectID::TYPE_PLAYER ) {
    return;
  }
  
  PlayerController* controller = dynamic_cast<PlayerController*>( iMaster->getController() );
  if ( controller != 0 ) {
    controller->update();
    if ( controller->shoot() != 0 ) {
      this->shoot();
    }
    // Update the mouse cursor and rotate the machine gun.
    Vec2D mousePos( mouse_x + Map::scrollX, mouse_y + Map::scrollY );
    fixed gunAngle = Utils::findVecAngle( this->position(), mousePos );
    this->angle( fixtoi(gunAngle) );
    
    mousePos.set( mouse_x, mouse_y );
    WarGlobals::pHud->crosshairPos( mousePos );
    if ( this->getCounter( KNoExitCounterIndex ) < 0 ) {
      if ( key[KEY_E] ) {
        WarGlobals::captureVehicle = true;
      }
    }
  }
}

void MachineGun::redraw( RedrawQueue* aQueue )
{
  BITMAP* gunGfx = GfxManager::findBitmap( GfxId::KAlienBigGun );
  RLE_SPRITE* supportGfx = GfxManager::findRleSprite( GfxId::KAlienBigGunSupport );
  
  Vec2D gunPos = this->position();
  gunPos -= Vec2D( Map::scrollX, Map::scrollY );
  gunPos -= Vec2D( supportGfx->w/2, supportGfx->h/2 );
  
  aQueue->add( RedrawQueue::PRI_BELOW_NORMAL, gunPos.intX(), gunPos.intY(),
               RedrawQueue::SPTYPE_RLE, supportGfx );
  aQueue->addRotatedSprite( RedrawQueue::PRI_NORMAL, gunPos.intX(), gunPos.intY(),
                            itofix( this->angle() ), gunGfx );
}

bool MachineGun::hitByBullet( Bullet* aBullet )
{
  if ( aBullet->iOwner != 0 ) {
    if ( aBullet->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
      return false;
    }
  }
  this->causeDamage( aBullet );
  this->makeSound( GameObject::SND_PAIN );
  WarGlobals::pPartManager->addSystem(
      new ParticleSparks(aBullet->iPosition, aBullet->velocity(), 12) );
  return true;
}

void MachineGun::makeSound( GameObject::SoundID aSoundId ) const
{
  switch ( aSoundId ) {
    case ( GameObject::SND_ATTACK ): {
      Weapon weapon( Weapon::W_MINIGUN );
      Sound::playSample( weapon.getSoundID(), false );
      break;
    }
    case ( GameObject::SND_PAIN ): {
      Sound::playSample( SMP_METALHIT, false );
      break;
    }
    case ( GameObject::SND_DIE ): {
      Sound::playSample( SMP_GRENADE, false );
      break;
    }
    default: {
      break;
    }
  }
}

void MachineGun::kill()
{
  this->state( GameObject::STATE_KILLED );
  this->hidden( true );
  this->makeSound( GameObject::SND_DIE );
  const Animation& anim = GameAnims::findAnimation(AnimId::KExplosionGrenade);
  AnimPlayer::spawn( anim, this->position(), 0 );
  if ( iMaster != 0 ) {
    iMaster->setHealth(0);
    iMaster->kill();
  }
  // And as the we explode, we emits some bullets
  float ac = 256.0 / 16.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );
    Bullet* b = BulletTable::createBullet( this, m_position, Bullet::ERifle );
    b->setVelocity( spdVec );
    b->iRange = 110;
    b->iDamage = 10;
    WarGlobals::pBulletManager->spawnBullet( b );
  }
}

ObjectID::Type MachineGun::objectType() const
{
  return ObjectID::TYPE_MACHINEGUN;
}

bool MachineGun::reloading() const
{
  if ( this->getCounter( KReloadCounterIndex ) < 0 ) {
    return false;
  }
  return true;
}

void MachineGun::capture( GameObject* aMaster )
{
  if ( aMaster == 0 ) {
    // We're about to get uncaptured
    if ( iMaster != 0 ) {
      iMaster->state( GameObject::STATE_LIVING );
      iMaster->hidden( false );
      Player* player = dynamic_cast<Player*>( iMaster );
      if ( player != 0 ) {
        player->setVehicle( 0 );
      }
      iMaster = 0;
      return;
    }
  } else {
    iMaster = aMaster;
    iMaster->state( GameObject::STATE_HIBERNATING );
    iMaster->hidden( true );
    iMaster->position( this->position() );
    Player* player = dynamic_cast<Player*>( iMaster );
    if ( player != 0 ) {
      player->setVehicle( this );
    }
    this->setCounter( KNoExitCounterIndex, 60 );
  }
}

void MachineGun::resurrect()
{
  this->state( GameObject::STATE_LIVING );
  this->setHealth( 100 );
  this->hidden(false);
}

bool MachineGun::canBeCaptured() const
{
  if ( iMaster == 0 ) {
    return true;
  }
  return false;
}

GameObject* MachineGun::getMaster() const
{
  return iMaster;
}


void MachineGun::shoot()
{
  if ( this->reloading() == true ) {
    return;
  }

  int noseIndex = (iActiveGunNose % 2 == 0 ) ? KGunNose1 : KGunNose2;
  iActiveGunNose += 1;
  
  Vec2D gunV( this->getCtrlPoint(noseIndex) );
  gunV += this->position();
  Bullet* bullet = BulletTable::createBullet( this, gunV, Bullet::EMinigun );
  bullet->iOwner = (iMaster == 0 ) ? this : iMaster;
  bullet->iDamage = Settings::floatObjProp(ObjectID::TYPE_MACHINEGUN, "bullet_damage:");
  WarGlobals::pBulletManager->spawnBullet( bullet );
  
  ParticleSystem* particles = new ParticleGunFlames( gunV, bullet->velocity(), 20, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( KReloadCounterIndex, 1 );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KMinigunShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  AnimPlayer::spawn( lightAnim, gunV, 0 );
}


};  // end of namespace
